
//Chris' WinGlide enhancements
//Copr. 1998, Chris Dohnal (cwdohnal@ucdavis.edu)

#include <windows.h>
#include <glide.h>
#include <ddraw.h>
#include "pointers.h"
#include "pcopy.h"
#include "main.h"
#include "dxdraw.h"
#include "errcode.h"
#include "msgbox.h"
#include "gamma.h"
#include "inifile.h"
#include "sysmenu.h"
#include "gammadlg.h"
#include "subclass.h"

BOOL OvlWGInitialize(HWND, DWORD, DWORD);
VOID OvlWGShutdown(VOID);
VOID OvlWGCopy3DfxFrameBuffer(VOID);

static BOOL gbDisplayErrorMessage = FALSE;
static char gszErrorMessage[256] = "";

static WNDPROC gpPrevWndProc = NULL;
static LRESULT CALLBACK SubclassWndProc(HWND, UINT, WPARAM, LPARAM);

static HBRUSH ghColorKeyBrush = NULL;

static LONG DisplayOverlay(HWND, LONG, LONG, DWORD, COLORREF);
static VOID MoveOverlayOnScreen(HWND);
static VOID DisplayCouldNotCreateOverlayErrorMessage(HRESULT);


BOOL OvlWGInitialize(HWND hWnd, DWORD dwWidth, DWORD dwHeight) {
	LONG lRet;
	BOOL bRet;
	DWORD dwMinStretch;
	DWORD dwClippingRecommendation;

	//Set defaults
	gbDisplayErrorMessage = FALSE;
	
	//Clear flags and pointers
	ghColorKeyBrush = NULL;


	//Initialize DirectDraw
	lRet = InitializeDirectDraw();
	if (lRet != EC_OK) {
		WinGlideErrorMessageBox(NULL, "Could not initialize DirectDraw.");

		goto initialization_failure;
	}

	//Set DirectDraw cooperative level
	lRet = SetNormalCooperativeLevel(hWnd);
	if (lRet != EC_OK) {
		WinGlideErrorMessageBox(NULL, "Could not set DirectDraw cooperative level.");

		goto initialization_failure;
	}
	
	
	//Clear certain variables before checking overlay capabilities
	dwClippingRecommendation = 0;

	//Check overlay capabilities
	lRet = CheckOverlayCaps(dwWidth, dwHeight, &dwMinStretch, &dwClippingRecommendation);

	if (lRet != EC_OK) {
		if (lRet == EC_MINIMUM_OVERLAY_STRETCH_FACTOR_TOO_LARGE) {
			int iRet;
			char szMessage[512];
			DWORD dwIntegerPart, dwDecimalPart;

			//Display an error message
			dwIntegerPart = dwMinStretch / 1000;
			dwDecimalPart = dwMinStretch % 1000;
			wsprintf(szMessage, "A base window size of %d and a minimum overlay stretch factor of %d.%d will prevent the overlay from being displayed even if the window is maximized.  Changing resolution or refresh rate might alter the minimum overlay stretch factor so that the overlay can be displayed.  Do you want to continue?", gdwScreenWidth, dwIntegerPart, dwDecimalPart);

			iRet = MessageBox(NULL, szMessage, "WinGlide", MB_YESNO | MB_DEFBUTTON2 | MB_ICONERROR);

			if (iRet != IDYES) {
				goto initialization_failure;
			}
		}
		else {
			switch (lRet) {
			case EC_COULD_NOT_GET_DIRECTDRAW_CAPS:
				WinGlideErrorMessageBox(NULL, "Could not get DirectDraw capabilities.");

				break;
			case EC_NO_OVERLAY_CAPS_DETECTED:
				WinGlideErrorMessageBox(NULL, "No overlay support detected.");

				break;

			default:
				//Show an error here just in case
				WinGlideErrorMessageBox(NULL, "Error checking DirectDraw capabilities.");
			}

			goto initialization_failure;
		}	
	}

	//See if overlay clipping capabilities should be automatically detected
	if (gbAutoDetectOverlayClipping == TRUE) {
		if (dwClippingRecommendation != 0) {
			//Clear the two clipping options in case they were set
			gdwOverlayFlags &= ~OVERLAY_ENABLE_COLOR_KEY_CLIPPING;
			gdwOverlayFlags &= ~OVERLAY_ENABLE_WINDOW_CLIPPING;

			//Set the recommended clipping flag
			gdwOverlayFlags |= dwClippingRecommendation;
		}
	}

	//Create the overlay
	lRet = CreateOverlay(dwWidth, dwHeight, gbUseDoubleBufferedOverlay, gdwBackBufferCount,
		gbUseFourCCSurface, gFourCC, gdwOverlayFlags, hWnd);
	if (lRet != EC_OK) {
		HRESULT hResult;

		//Look at the details of the error message
		switch (lRet) {
		case EC_COULD_NOT_CREATE_PRIMARY_SURFACE:
			WinGlideErrorMessageBox(NULL, "Could not create primary surface.");

			break;

		case EC_COULD_NOT_CREATE_OVERLAY_SURFACE:
			//Get the DxDraw error code
			hResult = GetDxDrawErrorCode();

			//Display an error message
			DisplayCouldNotCreateOverlayErrorMessage(hResult);

			break;

		case EC_COULD_NOT_GET_BACK_BUFFER:
			WinGlideErrorMessageBox(NULL, "Could not get back buffer overlay surface.");

			break;

		case EC_COULD_NOT_CREATE_CLIPPER:
			WinGlideErrorMessageBox(NULL, "Could not create clipper.");

			break;

		case EC_COULD_NOT_SET_CLIPPING_WINDOW:
			WinGlideErrorMessageBox(NULL, "Could not set clipping window.");

			break;

		case EC_COULD_NOT_ATTACH_CLIPPER:
			WinGlideErrorMessageBox(NULL, "Could not attach clipper.");

			break;

		default:
			//Show an error here just in case
			WinGlideErrorMessageBox(NULL, "Could not create overlay surface.");
		}

		goto initialization_failure;
	}

	//Create the color key brush
	ghColorKeyBrush = CreateSolidBrush(gcrColorKey);
	if (ghColorKeyBrush == NULL) {
		WinGlideErrorMessageBox(NULL, "Could not create color key brush.");

		goto initialization_failure;
	}

	//Create the gamma correction table if gamma correction is enabled
	//Also create the gamma correction table if a FourCC surface is being used
	if ((gbEnableGammaCorrection == TRUE) || (gbUseFourCCSurface == TRUE)) {
		//Allocate memory for the gamma correction table
		bRet = AllocateGammaCorrectionTable();
		if (bRet == FALSE) {
			ErrorLoadingWinGlideMessageBox(NULL, "Could not create gamma correction table.");

			goto initialization_failure;
		}

		if (gbUseFourCCSurface == TRUE) {
			//Make sure the only supported FourCC YUY2 is being used
			//TODO: maybe change how this works later
			if (gFourCC != mmioFOURCC('Y', 'U', 'Y', '2')) {
				char szFourCC[5];
				char szMessage[512];

				//Display an error message
				szFourCC[0] = (char)(gFourCC);
				szFourCC[1] = (char)(gFourCC >> 8);
				szFourCC[2] = (char)(gFourCC >> 16);
				szFourCC[3] = (char)(gFourCC >> 24);
				szFourCC[4] = 0;

				wsprintf(szMessage, "The FourCC %s is not supported by this version of WinGlide.\nThis version of WinGlide only supports the FourCC YUY2.", szFourCC);

				ErrorLoadingWinGlideMessageBox(NULL, szMessage);

				goto initialization_failure;
			}

			//Calculate the color conversion table
			if (gbEnableGammaCorrection == TRUE) {
				//Calculate the color conversion table
				CalculateColorConversionTable(gFourCC, gRedGamma, gGreenGamma, gBlueGamma);
			}
			else {
				//Calculate the color conversion table
				CalculateColorConversionTable(gFourCC, 1.0f, 1.0f, 1.0f);
			}
		}
		else {
			//Calculate the gamma correction table
			CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
		}
	}

	//Subclass the window and take care of other modifications to the window
	bRet = CWGSubclassWindow(hWnd, &gpPrevWndProc, SubclassWndProc, WINGLIDE_MODE_OVERLAY);
	if (bRet == FALSE) {
		ErrorLoadingWinGlideMessageBox(NULL, "Window subclassing error.");

		goto initialization_failure;
	}

	//Show the overlay
	DisplayOverlay(hWnd, dwWidth, dwHeight, gdwOverlayFlags, gcrColorKey);
	
	//Make sure the overlay is displayed entirely within the screen
	MoveOverlayOnScreen(hWnd);

	return TRUE;

	//Jump here if initialization failed
initialization_failure:

	//Clean up anything that was allocated
	OvlWGShutdown();

	return FALSE;
}

VOID OvlWGShutdown(VOID) {
	//Unsubclass the window if it was subclassed and clean up other modifications to the window
	CWGUnsubclassWindowIfSubclassed();
	
	//Destroy the color key brush if it was created
	if (ghColorKeyBrush != NULL) {
		DeleteObject(ghColorKeyBrush);
		ghColorKeyBrush = NULL;
	}
	
	//Destroy the overlay if it was created
	FreeOverlay();

	//Shutdown DirectDraw
	DestroyDirectDraw();

	//Unload the DirectDraw library if it was loaded
	UnloadDirectDraw();

	//Free the gamma correction table
	FreeGammaCorrectionTable();

	return;
}

VOID OvlWGCopy3DfxFrameBuffer(VOID) {
	FxBool fxbRet;
	GrLfbInfo_t grLFBInfo;
	BYTE *p3DfxLFBStart;
	LPVOID pSurface;
	DWORD dwSurfacePitch;
	BYTE *pOverlay;
	LONG lOverlayPitch;
	LONG lRet;
	BOOL bLockedOverlay;
	DWORD *pGammaTable;
	LONG lAlignedWidth;
	BOOL bRet;

	//Get the pointer to the gamma correction table
	if ((gbEnableGammaCorrection == TRUE) || (gbUseFourCCSurface == TRUE)) {
		pGammaTable = GetGammaCorrectionTablePointer();
		if (pGammaTable == NULL) {
			return;
		}
	}

	//Handle updating the gamma correction table if necessary
	bRet = CheckForNewGammaCorrectionValues(&gRedGamma, &gGreenGamma, &gBlueGamma);
	if (bRet == TRUE) {
		if (gbUseFourCCSurface == TRUE) {
			//Calculate the new color conversion table
			CalculateColorConversionTable(gFourCC, gRedGamma, gGreenGamma, gBlueGamma);
		}
		else {
			//Calculate the new gamma correction table
			CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
		}
	}

	//Lock the overlay surface
	lRet = LockOverlay(&pSurface, &dwSurfacePitch);
	if (lRet == EC_OK) {
		bLockedOverlay = TRUE;
		
		pOverlay = pSurface;
		lOverlayPitch = dwSurfacePitch;	
	}
	else {
		bLockedOverlay = FALSE;
	}

	if (bLockedOverlay == TRUE) {
		//Lock the 3Dfx LFB
		ZeroMemory(&grLFBInfo, sizeof(grLFBInfo));
		grLFBInfo.size = sizeof(grLFBInfo);
		fxbRet = _grLfbLock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER, 0,
			GR_ORIGIN_ANY, FXFALSE, &grLFBInfo);
		if (fxbRet == FXFALSE) {
			//Could not lock the 3Dfx LFB
			
			//Unlock the overlay surface
			UnlockOverlay();
			
			return;
		}

		//Setup the pointer to the 3Dfx LFB
		if (gOriginLocation == GR_ORIGIN_LOWER_LEFT) {
			p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
				(grLFBInfo.strideInBytes * (gdwScreenHeight - 1));
		}
		else {
			p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
				(grLFBInfo.strideInBytes * (gdwGlideScreenHeight - 1));
		}
		
		//Get the aligned width to copy
		lAlignedWidth = ((gdwScreenWidth + 3) & ~3) >> 2;
		
		
		//See if a FourCC surface is being used
		if (gbUseFourCCSurface == TRUE) {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXColorConvertPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
			else {
				ColorConvertPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
		}
		//See if gamma correction is enabled
		else if (gbEnableGammaCorrection == TRUE) {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXGammaPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
			else {
				GammaPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
		}
		else {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight);
			}
			else {
				FPUPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight);
			}
		}	

		//Unlock the overlay surface
		UnlockOverlay();

		//Unlock the 3Dfx LFB
		fxbRet = _grLfbUnlock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER);
		if (fxbRet == FXFALSE) {
			//Could not unlock the 3Dfx LFB
			return;
		}

		//See if double buffering is enabled
		if (gbUseDoubleBufferedOverlay == TRUE) {
			//Flip the overlay surface
			lRet = FlipOverlay();
		}
	}

	return;
}

static LRESULT CALLBACK SubclassWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	HDC hDC;
	HGDIOBJ hPrevObj;
	RECT rect;
	LRESULT lResult;
	static BOOL fNoSize;
	static RECT oldWindowRect;

	switch (message) {
	case WM_SYSCOMMAND:
		//See if system menu options are enabled
		if (gbSystemMenuOptions == TRUE) {
			BOOL bRet;

			//Handle WinGlide system menu options
			bRet = HandleWinGlideSystemMenuOptions(hWnd, wParam, lParam);

			//Break if the message was handled
			if (bRet == TRUE) {
				break;
			}

			//If the message was not handled, pass it on to the previous window procedure
		}

		//Send the WM_SYSCOMMAND message to the previous window procedure
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

	case WM_ENTERSIZEMOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		fNoSize = TRUE;

		//Save the old size of the window
		GetWindowRect(hWnd, &oldWindowRect);

		return lResult;

	case WM_EXITSIZEMOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		fNoSize = FALSE;

		//See if the size of the window was changed
		GetWindowRect(hWnd, &rect);

		if (((rect.bottom - rect.top) != (oldWindowRect.bottom - oldWindowRect.top)) ||
			((rect.right - rect.left) != (oldWindowRect.right - oldWindowRect.left))) {

			//Display the overlay
			GetClientRect(hWnd, &rect);
			DisplayOverlay(hWnd, rect.right - rect.left, rect.bottom - rect.top, gdwOverlayFlags, gcrColorKey);

			//Make sure the overlay is displayed entirely within the screen
			MoveOverlayOnScreen(hWnd);
		}

		return lResult;

	case WM_MOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Make sure the overlay is displayed entirely within the screen
		MoveOverlayOnScreen(hWnd);

		return lResult;

	case WM_SIZE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		if (fNoSize == TRUE) {
			break;
		}
		
		//Hide the overlay if the window is iconic
		if (IsIconic(hWnd)) {
			HideOverlay();
			break;
		}

		//Display the overlay
		DisplayOverlay(hWnd, LOWORD(lParam), HIWORD(lParam), gdwOverlayFlags, gcrColorKey);

		//Make sure the overlay is displayed entirely within the screen
		MoveOverlayOnScreen(hWnd);

		return lResult;

	case WM_DISPLAYCHANGE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Free and recreate the overlay
		FreeOverlay();
		CreateOverlay(gdwScreenWidth, gdwScreenHeight, gbUseDoubleBufferedOverlay, gdwBackBufferCount,
			gbUseFourCCSurface, gFourCC, gdwOverlayFlags, hWnd);

		//Display the overlay if the window is not iconic
		if (IsIconic(hWnd)) {
			break;
		}
		//Display the overlay
		GetClientRect(hWnd, &rect);
		DisplayOverlay(hWnd, rect.right - rect.left, rect.bottom - rect.top, gdwOverlayFlags, gcrColorKey);

		//Make sure the overlay is displayed entirely within the screen
		MoveOverlayOnScreen(hWnd);

		return lResult;

	case WM_PAINT:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//GetDC
		hDC = GetDC(hWnd);
		hPrevObj = SelectObject(hDC, ghColorKeyBrush);
		
		//Paint the background with the color key brush
		GetClientRect(hWnd, &rect);
		PatBlt(hDC, rect.left, rect.top, rect.right, rect.bottom, PATCOPY);
		
		//See if an error message should be displayed
		if (gbDisplayErrorMessage == TRUE) {
			DrawText(hDC, gszErrorMessage, -1, &rect, DT_CENTER | DT_SINGLELINE | DT_VCENTER);
		}

		//ReleaseDC
		SelectObject(hDC, hPrevObj);
		ReleaseDC(hWnd, hDC);

		//Validate the client area
		ValidateRect(hWnd, NULL);

		return lResult;

	case WM_DESTROY:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Unsubclass the window because it is being destroyed
		CWGUnsubclassWindowBecauseDestroyed();

		return lResult;

	default:
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
	}
	return 0;	
}

//Wrapper function for ShowOverlay
//Handles displaying the proper error message when something goes wrong
static LONG DisplayOverlay(HWND hWnd, LONG lWidth, LONG lHeight, DWORD dwOverlayFlags, COLORREF colorKey) {
	LONG lRet;
	HRESULT hResult;

	//Clear the error code just in case an error occurs but it does not get set
	SetDxDrawErrorCode(DD_OK);

	//Show the overlay
	lRet = ShowOverlay(lWidth, lHeight, dwOverlayFlags, colorKey);

	if (lRet == EC_OK) {
		//Make sure the error message is not displayed
		gbDisplayErrorMessage = FALSE;		
		lstrcpy(gszErrorMessage, "");
	}
	else {
		//Display an error message
		gbDisplayErrorMessage = TRUE;

		//Get the error code
		hResult = GetDxDrawErrorCode();
		
		//See what error code was returned
		switch (hResult) {
		case DDERR_GENERIC:
			lstrcpy(gszErrorMessage, "Generic error");
			break;
		case DDERR_HEIGHTALIGN:
			lstrcpy(gszErrorMessage, "Height alignment error");
			break;
		case DDERR_INVALIDOBJECT:
			lstrcpy(gszErrorMessage, "Invalid object");
			break;
		//Unsupported stretch factor
		case DDERR_INVALIDPARAMS:
		//Attempted to make overlay destination rectangle bigger than
		//the screen
		case DDERR_INVALIDRECT:
			lstrcpy(gszErrorMessage, "Invalid window size");
			break;
		case DDERR_INVALIDSURFACETYPE:
			lstrcpy(gszErrorMessage, "Invalid surface type");
			break;
		case DDERR_NOSTRETCHHW:
			lstrcpy(gszErrorMessage, "No stretch hardware");
			break;
		case DDERR_NOTAOVERLAYSURFACE:
			lstrcpy(gszErrorMessage, "Not an overlay surface");
			break;
		case DDERR_OUTOFCAPS:
			lstrcpy(gszErrorMessage, "Necessary hardware has already been allocated");
			break;
		case DDERR_SURFACELOST:
			lstrcpy(gszErrorMessage, "Surface lost");
			break;
		case DDERR_UNSUPPORTED:
			lstrcpy(gszErrorMessage, "Unsupported");
			break;
		case DDERR_XALIGN:
			lstrcpy(gszErrorMessage, "Horizontal alignment error");
			break;
		default:
			wsprintf(gszErrorMessage, "DisplayOverlay error (%lu)", hResult);
		}	
	}

	//Update the window
	InvalidateRect(hWnd, NULL, FALSE);
	UpdateWindow(hWnd);

	return lRet;
}

static VOID MoveOverlayOnScreen(HWND hWnd) {
	RECT rect;
	POINT point;
	int iScreenWidth, iScreenHeight;
	LONG lWidth, lHeight;

	//Find out where to move the upper left corner of the overlay to
	GetClientRect(hWnd, &rect);
	point.x = rect.left;
	point.y = rect.top;
	ClientToScreen(hWnd, &point);
	
	//Make sure the overlay fits on the screen
	iScreenWidth = GetSystemMetrics(SM_CXSCREEN);
	iScreenHeight = GetSystemMetrics(SM_CYSCREEN);
	lWidth = GetOverlayWidth();
	lHeight = GetOverlayHeight();
	if (point.x < 0) {
		point.x = 0;
	}
	if (point.y < 0) {
		point.y = 0;
	}
	if ((iScreenWidth - point.x) < lWidth) {
		point.x = iScreenWidth - lWidth;
	}
	if ((iScreenHeight - point.y) < lHeight) {
		point.y = iScreenHeight - lHeight;
	}
	
	//Move the overlay
	MoveOverlay(point.x, point.y);

	return;
}

static VOID DisplayCouldNotCreateOverlayErrorMessage(HRESULT hResult) {
	char szMessage[256];

	//Put the default first line in the message
	lstrcpy(szMessage, "Could not create overlay surface.\n");

	//See which error occurred
	switch (hResult) {
	case DDERR_INCOMPATIBLEPRIMARY:
		lstrcat(szMessage, "DDERR_INCOMPATIBLEPRIMARY\n");
		break;
	case DDERR_INVALIDCAPS:
		lstrcat(szMessage, "DDERR_INVALIDCAPS\n");
		break;
	case DDERR_INVALIDOBJECT:
		lstrcat(szMessage, "DDERR_INVALIDOBJECT\n");
		break;
	case DDERR_INVALIDPARAMS:
		lstrcat(szMessage, "DDERR_INVALIDPARAMS\n");
		break;
	case DDERR_INVALIDPIXELFORMAT:
		lstrcat(szMessage, "DDERR_INVALIDPIXELFORMAT\n");
		break;
	case DDERR_NOALPHAHW:
		lstrcat(szMessage, "DDERR_NOALPHAHW\n");
		break;
	case DDERR_NOCOOPERATIVELEVELSET:
		lstrcat(szMessage, "DDERR_NOCOOPERATIVELEVELSET\n");
		break;
	case DDERR_NODIRECTDRAWHW:
		lstrcat(szMessage, "DDERR_NODIRECTDRAWHW\n");
		break;
	case DDERR_NOEMULATION:
		lstrcat(szMessage, "DDERR_NOEMULATION\n");
		break;
	case DDERR_NOEXCLUSIVEMODE:
		lstrcat(szMessage, "DDERR_NOEXCLUSIVEMODE\n");
		break;
	case DDERR_NOFLIPHW:
		lstrcat(szMessage, "DDERR_NOFLIPHW\n");
		break;
	case DDERR_NOMIPMAPHW:
		lstrcat(szMessage, "DDERR_NOMIPMAPHW\n");
		break;
	case DDERR_NOOVERLAYHW:
		lstrcat(szMessage, "DDERR_NOOVERLAYHW\n");
		break;
	case DDERR_NOZBUFFERHW:
		lstrcat(szMessage, "DDERR_NOZBUFFERHW\n");
		break;
	case DDERR_OUTOFMEMORY:
		lstrcat(szMessage, "Not enough memory.\n");
		break;
	case DDERR_OUTOFVIDEOMEMORY:
		lstrcat(szMessage, "Not enough video memory.\n");
		break;
	case DDERR_PRIMARYSURFACEALREADYEXISTS:
		lstrcat(szMessage, "DDERR_PRIMARYSURFACEALREADYEXISTS\n");
		break;
	case DDERR_UNSUPPORTEDMODE:
		lstrcat(szMessage, "DDERR_UNSUPPORTEDMODE\n");
		break;
	default:
		;
	}

	//Display the error message
	WinGlideErrorMessageBox(NULL, szMessage);

	return;
}
